﻿/*
VERSION:		2.3
	
MIN USAGE:
	#include "map.as"
	makeMap( "map_mc" );		// creates a 20x15 map
	
MAX USAGE:
	#include "map.as"
	my_map = makeMap( "map_mc", this, 42, 9, 9 );		// creates a 9x9 map at depth 42
	
FUNCTIONS:
	makeMap()		Creates the map system.
	load()				Loads, converts, resets, and draws a map, based on an XML file.  It also loads the collision data.
	drawObj()		Resets & draws a map, based on the data provided.  It also imports the collision data.
	setTile()			Draws a tile.
	reset()			Clears the chipset & all layers.
	loadChipset()	Loads a chipset.  (file or linkage)
	scroll()			Moves the map to center on a movieClip inside of it.
	
EVENTS:
	onLoad()			Runs after a chipset is loaded.
	
VARIABLES:
	width					Width of the map, in tiles.
	height				Height of the map, in tiles.
	collision_array		2D array containing collision values.
	chipset_pic			Bitmap containing chipset palette.
	layer0_pic			Image containing all tiles on layer 0.		(maps can have any amount of layers)
	layer0_mc			Movieclip displaying layer0_pic.
	layer1_pic			Image containing all tiles on layer 1.
	layer1_mc			Movieclip displaying layer1_pic.
	
DEPENDANCIES:
	addCollisionMethods.as
	nextDepth.as
	readXml.as
	
NOTE:
	XML data should be saved in a <map> tag, like so:
	writeXml( myData, myXml, "map" );
	
NOTE:
	Layers are assigned depths in incriments of 10
		0 = 0
		1 = 10
		2 = 20
	This allows movieClips to be inserted between them.
	
NOTE:
	chipsets can be either external files, or bitmaps with linkage.
	
TO USE LINKAGE INSTEAD OF EXTERNAL FILES:
	You can pull this off by copying the xml data into a VERY long string,
	passing the string to a new XML object,
	and passing the XML object to the map's drawObj() function.
	
	rawData = '<map>~~';
	rawData += '~~</map>';				(with larger maps, you may need to split the string like this to avoid weird errors)  (my rule of thumb is to split the string when it reaches half a page long)
	data = new XML( rawData );
	my_map.drawObj( data );
	
	Typically, I put the above code inside of a movieClip and load a new map by placing the movieClip with attachMovie()
*/
makeMap = function( instanceName, target_mc, depth, width, height )
{
	#include "nextDepth.as"
	
	// resolve optional parameters
	var target_mc = (target_mc) ? target_mc : this;
	var depth = (depth != undefined) ? depth : nextDepth(target_mc);
	var width = (width) ? width : 20;
	var height = (height) ? height : 15;
	
	// create container movieClip
	var _this = target_mc.createEmptyMovieClip( instanceName, depth );
	
	// define internal variables
	_this.width = width;
	_this.height = height;
	
	// create empty chipset
	_this.chipset_pic = new flash.display.BitmapData( 480, 256, true, 0 );
	
	// create empty collision array
	_this.collision_array = new Array();
	for(var x=0; x<width; x++)
	{
		_this.collision_array[x] = new Array();
		for(var y=0; y<height; y++)
		{
			_this.collision_array[x][y] = 0;
		}// for y
	}// for x
	#include "addCollisionMethods.as"
	addCollisionMethods( _this.collision_array );
	
	
	
	
	
	// ________________________________________________________
	// 	FUNCTIONS
	
	_this.reset = function( width, height )
	{
		// remove movieClips
		// remove images
		for (var nam in _this)
		{
			var thisObj = _this[nam];
			// remove layers
			if( typeof(thisObj) == "movieclip" )
			{
				if (nam.slice(0, 5) == "layer")
				{
					thisObj.removeMovieClip();
				}
			}
			// remove layer images  &  chipset
			else if( thisObj instanceof flash.display.BitmapData )
			{
				if (nam.slice(0, 5) == "layer"  ||
					nam == "chipset_pic")
				{
					_this[nam].dispose();
					delete _this[nam];		// delete original reference
				}
			}
		}// for-in:  _this
		
		// reset variables
		_this.width = (width) ? width : _this.width;		// if new size isn't specified, retain existing size
		_this.height = (height) ? height : _this.height;
		// reset chipset image
		_this.chipset_pic = new flash.display.BitmapData( 480, 256, true, 0 );
		//var destArea = new flash.geom.Rectangle( 0, 0, 480, 256 );
		//_this.chipset_pic.fillRect( destArea, 0 );
		
		// create empty collision array
		_this.collision_array = new Array();
		for(var x=0; x<_this.width; x++)
		{
			_this.collision_array[x] = new Array();
			for(var y=0; y<_this.height; y++)
			{
				_this.collision_array[x][y] = 0;
			}// for y
		}// for x
		#include "addCollisionMethods.as"
		addCollisionMethods( _this.collision_array );
	}// reset()
	
	
	
	
	
	_this.setTile = function(x, y, id, layer)
	{
		// resolve optional parameters
		var layer = (layer != undefined) ? layer : 0;
		
		// determine copy coordinates, based on id
		var chipWidth = 30;
		var xChip = id % chipWidth;
		var yChip = Math.floor( id / chipWidth );
		var copyArea = new flash.geom.Rectangle(
											xChip*16,
											yChip*16,
											16,
											16);
		
		// determine paste coordinates
		var pastePoint = new flash.geom.Point(
											  x*16,
											  y*16);	
		
		// create layer if neccessary
		var layerName = "layer"+layer+"_mc";			// layer2_mc
		var imageName = "layer"+layer+"_pic";		// layer2_pic
		if( _this[layerName] == undefined )
		{// if:  layer doesn't exist
			// image
			var thisImage = _this[imageName] = new flash.display.BitmapData(
															_this.width*16,
															_this.height*16,
															true,
															0);
			// movieClip
			_this.createEmptyMovieClip( layerName, layer*10 );
			_this[layerName].attachBitmap( thisImage, 0, true );
		}// if:  layer doesn't exist
		
		// copy the tile image to the layer
		_this[imageName].copyPixels( _this.chipset_pic, copyArea, pastePoint );
	}// setTile()
	
	
	
	
	
	_this.loadChipset = function( fileName )
	{
		// clear image
		var fillArea = new flash.geom.Rectangle(0,0,480,256);
		_this.chipset_pic.fillRect( fillArea, 0 );
		
		// load image
		var newChip_pic = flash.display.BitmapData.loadBitmap( fileName );
		if( newChip_pic.width > -1 )
		{// linkage succeeded
			_this.chipset_pic = newChip_pic.clone();
			newChip_pic.dispose();
			delete newChip_pic;
			
			// call external function
			_this.onLoad();		// externally-defined function
		}// if:  linkage succeeded
		else
		{// linkage failed
			_this.createEmptyMovieClip( "load_mc", 99 );
			_this.loader = new MovieClipLoader();
			_this.loader.loadClip( fileName, _this.load_mc );
			
			
			_this.loader.onLoadInit = function( load_mc )
			{
				// copy loaded image
				// // snapshot
				var newChip_pic = new flash.display.BitmapData( 480, 256, true, 0 );
				newChip_pic.draw( _this.load_mc );
				// // copy
				_this.chipset_pic = newChip_pic.clone();
				newChip_pic.dispose();
				delete newChip_pic;
				
				// call external function
				_this.onLoad();		// externally-defined function
				
				// // clean-up
				_this.loader.unloadClip( load_mc );		// removes load_mc
				delete _this.loader;
			}// chipset loaded()
		}// if:  linkage failed
	}// loadChipset()
	
	
	
	
	
	_this.drawObj = function( data_obj )
	{
		// reset
		_this.reset();
		
		// width
		_this.width = data_obj.width;
		
		// height
		_this.height = data_obj.height;
		
		// collision
		if( data_obj.collision != undefined)
		{// if:  collision data exists
			_this.collision_array = new Array();
			for(var x=0; x<data_obj.width; x++)
			{
				_this.collision_array[x] = new Array();
				for(var y=0; y<data_obj.height; y++)
				{
					_this.collision_array[x][y] = data_obj.collision[x][y];
				}// for y
			}// for x
			#include "addCollisionMethods.as"
			addCollisionMethods( _this.collision_array );
		}// if:  collision data exists
		
		
		
		// chipset_pic
		// // clear image
		var fillArea = new flash.geom.Rectangle(0,0,480,256);
		_this.chipset_pic.fillRect( fillArea, 0 );
		
		// load image
		var newChip_pic = flash.display.BitmapData.loadBitmap( data_obj.chipset );
		if( newChip_pic.width > -1 )
		{// linkage succeeded
			_this.chipset_pic = newChip_pic.clone();
			newChip_pic.dispose();
			delete newChip_pic;
			
			
			
			// draw layers  (setTile)
			for (lay=0; lay<data_obj.layers.length; lay++)
			{
				for (var x=0; x<_this.width; x++)
				{
					for (var y=0; y<_this.height; y++)
					{
						var id = data_obj.layers[lay][x][y];
						_this.setTile( x, y, id, lay );
					}// for:  width
				}// for:  height
			}// for:  layers
			
			
			
			// call external function
			_this.onLoad();		// externally-defined function
			
			
			// clean-up
			delete _this.loadData;		// In case load() called this function, then clean up after load() as well.
		}// if:  linkage succeeded
		else
		{// linkage failed
			_this.createEmptyMovieClip( "load_mc", 99 );
			_this.loader = new MovieClipLoader();
			_this.loader.loadClip( data_obj.chipset, _this.load_mc );
			

			_this.loader.onLoadInit = function( load_mc )
			{
				// // copy loaded image
				// // // snapshot
				var newChip_pic = new flash.display.BitmapData( 480, 256, true, 0 );
				newChip_pic.draw( _this.load_mc );
				// // // copy
				var copyArea = new flash.geom.Rectangle( 0, 0, 480, 256 );
				_this.chipset_pic.copyPixels( newChip_pic, copyArea, {x:0,y:0} );
				// // // clean-up
				_this.loader.unloadClip( _this.load_mc );
				
				
				
				// draw layers  (setTile)
				for (lay=0; lay<data_obj.layers.length; lay++)
				{
					for (var x=0; x<_this.width; x++)
					{
						for (var y=0; y<_this.height; y++)
						{
							var id = data_obj.layers[lay][x][y];
							_this.setTile( x, y, id, lay );
						}// for:  width
					}// for:  height
				}// for:  layers
				
				
				// call external function
				_this.onLoad();		// externally-defined function
				
				
				// clean-up
				delete _this.loadData;		// In case load() called this function, then clean up after load() as well.
				delete _this.loader;
			}// chipset loaded()
		}// if:  linkage failed
	}// drawObj()
	
	
	
	
	
	_this.load = function( fileName )
	{
		// load XML
		XML.prototype.ignoreWhite = true;
		// attempt linkage method
		_this.attachMovie( fileName, "loadMap_mc", 98 );
		if(_this.loadMap_mc)
		{// linkage
			_this.loadMap_mc.onLoad = function( data )
			{
				// copy xml data
				_this.load_xml = data;
				
				// convert XML
				_this.loadData = new Object();
				#include "readXml.as"
				readXml( _this.load_xml, _this.loadData );
				
				// pass object to drawObj()
				_this.drawObj( _this.loadData.map );
				
				// clean-up
				loadMap_mc.removeMovieClip();
			}// onLoad()
		}// if:  use linkage
		else
		{// external file
			// if fail
			_this.load_xml = new XML();
			_this.load_xml.load( fileName );
			_this.load_xml.onLoad = function( success )
			{
				#include "readXml.as"
				if( success )
				{
					// convert XML
					_this.loadData = new Object();
					readXml( _this.load_xml, _this.loadData );
					
					// pass object to drawObj()
					_this.drawObj( _this.loadData.map );
					
				}// if:  success
			}// onLoad()
		}// if:  use external file
	}// load()
	
	
	
	_this.scroll = function( x, y, screenWidth, screenHeight, smoothness)
	{
		// resolve optional parameters
		var screenWidth = (screenWidth) ? screenWidth : 320;
		var screenHeight = (screenHeight) ? screenHeight : 240;
		var smoothness = (smoothness) ? smoothness : 1;
		// calculate view screen center
		var halfWidth = screenWidth / 2;
		var halfHeight = screenHeight / 2;
		// get the map's pixel width
		var mapWidth = _this.width * 16;
		var mapHeight = _this.height * 16;
		
		
		// horz
		if( x < halfWidth)
		{
			// far-left
			var xTarg = 0;
		}
		else if( x > mapWidth-halfWidth)
		{
			// far-right
			var xTarg = -mapWidth+screenWidth;
		}
		else
		{
			// scroll
			var xTarg = -x + halfWidth;
		}
		
		
		// vert
		if( y < halfHeight)
		{
			// top
			var yTarg = 0;
		}
		else if( y > mapHeight-halfHeight)
		{
			// bottom
			var yTarg = -mapHeight+screenHeight;
		}
		else
		{
			// scroll
			var yTarg = -y + halfHeight;
		}
		
		
		// apply target position
		_this._x += (xTarg-_this._x) * smoothness;
		_this._y += (yTarg-_this._y) * smoothness;
	}// scroll()
	
	
	
	// ________________________________________________________
	
	
	
	// return reference to this map
	return _this;
}// makeMap()